// version 1

import * as utils from "./js/utils.js";

// Cache names with version numbers (increment when updating cached resources)
const STATIC_CACHE_NAME = 'static-v1';
const DYN_CACHE_NAME = `dynamic-v1`;

// App shell files that need to be cached for offline functionality
// These are the core files needed for the app to work offline
const STATIC_APP_SHELL = [
    '/',
    '/index.html',
    '/css/stylesheet.css',
    '/js/app.js',
    '/js/utils.js',
    '/manifest.json',
    '/assets/icons/icon-192.png',
    '/assets/icons/icon-512.png',
    '/assets/icons/icon-180.png',
    '/assets/icons/icon-144.png',
    '/assets/icons/icon-maskable-512.png',
    '/assets/icons/icon-maskable-192.png',
    '/assets/icons/notification-badge.png',
    '/assets/icons/favicon.png'
].map((url) => new URL(url, self.location.href)); // Convert to absolute URLs

// How long cached data stays fresh before needing update (24 hours)
const STALE_TIME = 24 * 60 * 60 * 1000;

// Maximum number of requests to store in the service-worker cache (you can safely increase this)
const MAX_CACHE_SIZE = 240;

// Register service worker event listeners: install, activate, fetch
self.oninstall = (evt) => evt.waitUntil(install_service());
self.onactivate = (evt) => evt.waitUntil(activate_service());
self.onfetch = (evt) => evt.respondWith(fetch_service(evt));

// Install event listener
async function install_service() {
    try {
        console.log("Installing service worker...");

        // Open/Create the static cache and store all app shell files
        const cache = await caches.open(STATIC_CACHE_NAME);
        await cache.addAll(STATIC_APP_SHELL);

        // Skip waiting phase to activate immediately
        await self.skipWaiting();

        console.log("Service Worker installed successfully.");
    } catch (error) {
        console.error("Error during SW installation:", error.message);
        // Re-throw to fail installation if critical files can't be cached
        throw error;
    }
}

// Activate service worker and clean up old caches
async function activate_service() {
    try {
        console.log("Activating service worker...");

        // Get all existing cache names
        const key_list = await caches.keys();

        // Find old cache versions that should be deleted
        const keys_to_delete = key_list.filter(key =>
            key !== STATIC_CACHE_NAME && key !== DYN_CACHE_NAME
        );

        // Delete all old caches in parallel
        await Promise.all(keys_to_delete.map(key => caches.delete(key)));

        // Take control of all existing clients immediately
        await self.clients.claim();

        console.log("Service Worker activated successfully.");
    } catch (error) {
        console.error("Error during SW activation:", error.message);
        throw error;
    }
}

// Check if a URL represents a static resource that should be cached long-term
function is_static(url) {
    // Check if the URL matches any file in our app shell
    // OR
    // Also treat common static file extensions as static resources
    return (
        STATIC_APP_SHELL.some((asset) => asset.pathname === url.pathname)
        ||
        (/\.(css|js|png|jpg|svg|ico)$/.test(url.pathname))
    );
}

// Validate if an API response is ok and not empty
async function validate_response(response) {
    // Opaque responses (from CORS) can't be inspected but may still be valid
    // OR
    // If the response is ok and has a non-zero body size
    return (
        (response.type === 'opaque')
        ||
        Boolean(response?.ok && (await response.clone().blob()).size > 0)
    );
}

// Keep cache size under control by removing older entries when limit is reached
async function limit_cache_size(cache) {
    const cached_requests = await cache.keys();

    // If cache is getting full, remove older half of entries
    if (cached_requests?.length >= MAX_CACHE_SIZE) {
        const older_requests = cached_requests.slice(0, MAX_CACHE_SIZE / 2);

        try {
            // Delete older entries in parallel with Promise.all() for better performance
            await Promise.all(older_requests.map(req => cache.delete(req)));
        } catch (error) {
            console.error('Error deleting old cache entries:', error);
        }
    }
}

// Store a response in cache with timestamp and size management
async function store_in_cache(url_string, response, cache_name) {
    // Validate the response before caching
    if (!(await validate_response(response))) {
        throw new Error(`Invalid or empty response for request: ${url_string}`);
    }

    try {
        const cache = await caches.open(cache_name);

        // Add timestamp to response headers for freshness tracking (if possible)
        if (response.type !== 'opaque') {
            const headers = new Headers(response.headers);
            headers.append("X-last-fetched", Date.now().toString());

            // Create new response with updated headers
            response = new Response(response.body, {
                status: response.status,
                statusText: response.statusText,
                headers: headers
            });
        }

        // Check Cache size and delete older entries before adding new one
        await limit_cache_size(cache);

        // Add the new one
        await cache.put(url_string, response);
    } catch (error) {
        console.error(`Error caching response for ${url_string}: ${error.message}`);
    }
}

async function cache_first_strategy(request, cache_name) {
    // Try to find the resource in cache first
    const cached_resp = await caches.match(request.url);
    if (cached_resp) {
        return cached_resp;
    }

    // If not in cache, fetch from the network and cache its clone for future use
    // Note: If an error occurs, it will be caught in the fetch event handler
    let fresh_resp;
    fresh_resp = await fetch(request);
    await store_in_cache(request.url, fresh_resp.clone(), cache_name);

    return fresh_resp;
}

// Network First Strategy: Try network first, fallback to cache if network fails
async function network_first_strategy(request, cache_name) {
    try {
        // try network first
        const fresh_resp = await fetch(request);
        await store_in_cache(request.url, fresh_resp.clone(), cache_name);
        return fresh_resp;
    } catch (error) {
        // Network failed - check if we have a cached version
        const cache_resp = await caches.match(request.url);
        if (!cache_resp) {
            throw error; // No cached fallback available
        }
        return cache_resp;
    }
}

// Stale-While-Revalidate Strategy: Serve cached content if fresh, otherwise fetch new
async function api_strategy(request) {
    let fetch_fresh_content = true;
    let fresh_resp;

    // Check if we have a cached version
    const cached_resp = await caches.match(request.url);
    if (cached_resp) {
        // Check if the cached content is still fresh (within STALE_TIME)
        const last_fetched = cached_resp.headers.get("X-last-fetched");
        fetch_fresh_content = !last_fetched || (Date.now() - Number(last_fetched) > STALE_TIME);

        // If fetching fresh content is not needed as per our requirement
        if (!fetch_fresh_content) {
            return cached_resp;
        }
    }

    // If we need fresh content, try to fetch from network
    if (fetch_fresh_content) {
        try {
            fresh_resp = await fetch(request);
            await store_in_cache(request.url, fresh_resp.clone(), DYN_CACHE_NAME);
            return fresh_resp;
        } catch (error) {
            // Network failed - use cached version if available, else throw error
            if (!cached_resp) {
                throw error;
            }
            return cached_resp;
        }
    }
}

// Fetch event handler: intercepts all network requests and applies caching strategies
async function fetch_service(evt) {
    let response;
    let request = evt.request;
    let url = new URL(request.url);
    let url_string = url.href;

    try {
        // Let other methods (POST, PUT, DELETE) go through normally
        if (request.method !== "GET") {
            return fetch(request);
        }

        // Route request to appropriate caching strategy based on resource type
        if (is_static(url)) {
            // Static resources (CSS, JS, images)
            response = await cache_first_strategy(request, STATIC_CACHE_NAME);
        }
        else if (url_string.startsWith(utils.BASE_API_URL)) {
            // News API calls
            response = await api_strategy(request);
        }
        else {
            // Other Dynamic content
            response = await network_first_strategy(request, DYN_CACHE_NAME);
        }
    } catch (error) {
        console.error(`Error in fetching ${url_string}: ${error.message}`);

        // If content cannot be served from either cache or network
        response = new Response(null, {
            status: 503,
            statusText: "Service Unavailable"
        });
    }

    return response;
}
