// Base URL for the NewsData.io API endpoint
export const BASE_API_URL = "https://newsdata.io/api/1/latest";

// API key for authentication (replace with your own; avoid hardcoding like this in production for security)
export const API_KEY = "";

// Number of articles to fetch per interest category
export const SIZE_PER_REQUEST = 5;

// Cookie expiration time in milliseconds (setting this to 365 days)
export const COOKIE_LIFE = 365 * 24 * 60 * 60 * 1000;

// Map of available news categories with labels and emojis for UI display
export const ALL_CATEGORIES = new Map([
    ["top", {label:"Trending (Default)", emoji:"📈"}],
    ["technology", {label:"Technology", emoji:"💻"}],
    ["health", {label:"Health", emoji:"🏥"}],
    ["business", {label:"Business", emoji:"💼"}],
    ["sports", {label:"Sports", emoji:"⚽"}],
    ["entertainment", {label:"Entertainment", emoji:"🎬"}],
    ["science", {label:"Science", emoji:"🔬"}],
    ["politics", {label:"Politics", emoji:"🏛️"}],
    ["tourism", {label:"Travel", emoji:"✈️"}],
    ["food", {label:"Food", emoji:"🍔"}],
    ["lifestyle", {label:"Lifestyle", emoji:"👗"}],
    ["environment", {label:"Environment", emoji:"🌱"}]
]);

// Map of supported languages for user preferences
export const ALL_LANGUAGES = new Map([
    ["en", "English (Default)"],
    ["es", "Español"],
    ["fr", "Français"],
    ["de", "Deutsch"],
    ["it", "Italiano"],
    ["pt", "Português"],
    ["hi", "हिन्दी"],
    ["bn", "বাংলা"],
    ["ta", "தமிழ்"],
    ["zh", "中文"],
    ["ja", "日本語"],
    ["ko", "한국어"],
    ["ar", "العربية"],
    ["sw", "Kiswahili"]
]);

// Default user preferences to fall back to if none are set
export const DEFAULT_PREFERENCES = {
    interests: ["top"],
    language: "en",
    notifications: false
};

// Save user preferences to browser cookies with expiration date
export async function save_user_preferences(preferences) {
    try {
        await cookieStore.set({ 
            name: "user-preferences", 
            value: JSON.stringify(preferences), 
            expires: (new Date(Date.now() + COOKIE_LIFE)) 
        });
    } catch (error) {
        console.error(`Error saving user preferences to cookies: ${error.message}`);
    }
}

// Retrieve user preferences from cookies, fallback to defaults if not found
export async function get_user_preferences() {
    let preferences;
    try {
        preferences = await cookieStore.get("user-preferences");
        preferences = JSON.parse(preferences.value);
    } catch (error) {
        console.error(`Error retrieving user preferences from cookies: ${error.message}`);
        preferences = DEFAULT_PREFERENCES;
    }
    return preferences;
}

// Fetch news articles for multiple categories and return them as a combined array
export async function fetch_news({ categories = ["top"], language = "en" } = {}) {
    const results = [];
    
    // Loop through each category to fetch articles 
    for (const cat of categories) {
        try {
            // Build the complete API URL with category and language parameters
            const url = construct_url({ category: cat, language: language });
            
            // Make the API request
            const response = await fetch(url);
            
            // Check if response contains JSON data
            if (response.headers.get("content-type")?.includes("application/json")) {
                const data = await response.json();
                
                // Process successful API responses with articles
                if (data.status === "success" && data.results?.length > 0) {
                    // Clean up the response data to keep only needed fields
                    const trunc_results = trunc_response(data.results);
                    results.push(...trunc_results); // Add articles to combined results
                    
                    // Update timestamp for when news was last fetched
                    await cookieStore.set({ 
                        name: "last-updated-news", 
                        value: new Date().toLocaleString() 
                    });
                }
            }
        } catch (error) {   
            console.error(error.message);
        }
    }
    
    // Return all articles from all categories as one array
    return results;
}

// Build the complete API URL with query parameters
function construct_url(opts = {}) {
    const query = new URLSearchParams({ 
        apikey: API_KEY, 
        size: SIZE_PER_REQUEST,
        prioritydomain: "top",
        image: 1,
        removeduplicate: 1,
        country: "wo",
        ...opts 
    }).toString();
    return `${BASE_API_URL}?${query}`;
}

// Extract only the needed fields from the raw API response
function trunc_response(data_results) {
    const trunc_results = [];
    
    for (const article of data_results) {
        const trunc_article = {
            title: article.title,
            link: article.link,
            author: article.creator?.join(', ') ?? "unknown",
            published: (article.pubDate && article.pubDateTZ) ? 
                (`${date_format(article.pubDate)} ${article.pubDateTZ}`) : 
                (date_format(article.pubDate) ?? "unknown"),
            description: article.description ?? "",
            category: article.category?.join(", "),
            image_url: article.image_url ?? "",
            source_icon: article.source_icon ?? "",
            source_id: article.source_id ?? ""
        };
        trunc_results.push(trunc_article);
    }
    
    return trunc_results;
}

// Format date strings into readable format based on user's locale
function date_format(date_string) {
    const d = date_string ? new Date(date_string) : new Date();
    if (isNaN(d)) 
        return null;
        
    return d.toLocaleDateString(undefined, { 
        weekday: 'short', 
        day: 'numeric', 
        month: 'short', 
        year: 'numeric' 
    });
}

