import * as utils from "./utils.js";

// Populate the language selection dropdown using ALL_LANGUAGES Map from utils.js
function create_lang_options_ui() {
    const select = document.getElementById("language-options");
    
    // Loop through and create each language option
    for (const [language, label] of utils.ALL_LANGUAGES) {
        const option = document.createElement("option");

        // Set the value of the option to the language code (e.g., "en" for English)
        option.value = language;

        // Set the label
        option.textContent = label;  
        
        // Add the option to the select dropdown
        select.appendChild(option);
    }
}

// Populate the interests checkboxes using ALL_CATEGORIES Map from utils.js
function create_interests_checkboxes_ui() {
    const div = document.getElementById("interests-checkboxes");

    // Loop through and create each interest-category checkbox
    for (const [interests, { label, emoji }] of utils.ALL_CATEGORIES) {
        // Create a label element to wrap the checkbox and text
        const label_elem = document.createElement("label");
        
        // Set the inner HTML to include checkbox input and display text with emoji
        label_elem.innerHTML = `<input type="checkbox" name="interests" value="${interests}">${label} <span class="emoji">${emoji}</span>`;
        
        // Add the label (with checkbox) to the container div
        div.appendChild(label_elem);
    }
}

// Display an alert message to the user with auto-close functionality
function show_alert(alert_msg) {
    if (alert_msg.trim().length > 0) {
        document.getElementById("alert-box").classList.remove("hidden");
        document.getElementById("alert-message").textContent = alert_msg;
        setTimeout(close_alert, 10000); // Auto-close after 10 seconds
    }
}

// Hide the alert box
function close_alert() {
    document.getElementById("alert-box").classList.add("hidden");
}

// Handle form submission: validate input, request permissions, and save preferences
async function form_submit(evt) {
    evt.preventDefault();
    const data = new FormData(this);

    // Get the selected language (single value)
    const language = data.get("language");

    // Get all selected interests (multiple checkboxes can be selected)
    const interests = data.getAll("interests");

    // Validate that at least one interest is selected
    if (interests.length === 0) {
        show_alert("Select at least one interest. Otherwise, the default will be used.");

        // Use default "top" interest category if none selected (top is trending news)
        interests.push("top"); 
    }

    // Check if notifications are enabled (checkbox value is "1" when checked)
    let notifications = data.get("notifications") === "1";

    // If user wants notifications, prompt browser permission
    if (notifications) {
        const permission = await Notification.requestPermission();

        // Enable notifications only if permission to the browser is granted
        notifications = (permission === "granted");
    }

    // Create preferences object to save
    const preferences = {
        interests: interests,
        language: language,
        notifications: notifications
    };

    // Save preferences
    await utils.save_user_preferences(preferences);
    
    // Update news display section with the new preferences
    await update_news(preferences);
}

// Update the UI form fields based on saved user preferences
function update_ui(preferences) {
    document.getElementById("language-options").value = preferences.language;
    document.querySelectorAll("input[name='interests']").forEach(checkbox => {
        checkbox.checked = preferences.interests.includes(checkbox.value);
    });
    document.querySelector("input[name='notifications']").checked = preferences.notifications;
}

// Fetch news articles and update the news display section
async function update_news(preferences) {
    const news_list = document.getElementById("news-list");
    
    // Store current content as backup in case of errors
    const before_update = news_list.innerHTML;
    
    // Show loading message while fetching
    news_list.innerHTML = "<p>Loading latest news articles...</p>";

    try {
        // Fetch articles based on user preferences
        const articles = await utils.fetch_news({ 
            categories: preferences.interests, 
            language: preferences.language 
        });
        
        if (Array.isArray(articles) && articles.length > 0) {
            // Build HTML content from articles
            const news_fragment = get_news_fragment(articles);
            
            // Replace loading message with actual content
            news_list.innerHTML = "";
            news_list.appendChild(news_fragment);
            
            // Update the "last updated" timestamp display
            const last_updated = (await cookieStore.get("last-updated-news"))?.value;
            document.getElementById("last-updated").textContent = last_updated || "Unknown";
        }
    } catch (error) {
        // Show error to user and restore previous content
        show_alert(error.message);
        news_list.innerHTML = before_update;
    }
}

// Build HTML elements for news articles using DocumentFragment for performance
function get_news_fragment(news_articles) {
    // Validate that we have articles to display
    if (!news_articles.length) {
        throw new Error("No articles found for your selected interests. Try different interests or check back later.");
    }
    
    // Use DocumentFragment to build all articles efficiently (avoids multiple DOM reflows)
    const fragment = document.createDocumentFragment();
    
    for (const article of news_articles) {
        // Create clickable article container
        const article_link = document.createElement("a");
        article_link.href = article.link;
        article_link.target = "_blank"; // Open in new tab
        article_link.classList.add("news-link");

        // Build optional HTML for source icon and article image
        const source_icon_html = article.source_icon ? 
            `<img src="${article.source_icon}" class="source-icon" alt="${article.source_id} icon">` : '';
        const image_html = article.image_url ? 
            `<img src="${article.image_url}" class="news-image" alt="${article.title}">` : '';

        // Construct the complete article HTML structure
        article_link.innerHTML = `
            <h3>${source_icon_html}${article.title}</h3>
            <div class="news-category">
                <span><em>${article.author || "Unknown"} on ${article.published}</em></span>
                <span><strong>Category:</strong> ${article.category}</span>
            </div>
            ${image_html}
            <p>${article.description}</p>
            <div class="news-source">
                <span>Source: ${article.source_id || 'Unknown'}</span>
            </div>
        `;
        
        fragment.appendChild(article_link);
    }
    return fragment;
}

// Register the service worker
async function service_worker_registration() {
    // Check if the browser supports service workers
    if ('serviceWorker' in navigator) {
        try {
            console.log("Registering service worker...");
            
            // Register the service worker file (sw.js)
            await navigator.serviceWorker.register("./sw.js", { 
                scope: "./",      // Control all pages in this directory
                type: "module"    // Allow ES6 imports in service worker
            });

            console.log("Service Worker registered successfully.");
        } catch (error) {
            // Service worker registration failed (maybe browser doesn't support it)
            console.error('Service Worker registration failed:', error.message);
        }
    }
    // If service workers aren't supported, the app still works without offline features
}

// This is the entry point - runs when the page loads
async function init() {
    create_lang_options_ui();
    create_interests_checkboxes_ui();
    const preferences = await utils.get_user_preferences();
    update_ui(preferences);
    await update_news(preferences);
    await service_worker_registration();
}

// Register event listeners: form submission, alert's close button, and DOMContentLoaded
document.getElementById("user-preferences-form").addEventListener("submit", form_submit);
document.getElementById("alert-close-btn").addEventListener("click", close_alert);
document.addEventListener("DOMContentLoaded", init);